/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'AbstractFigure.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing;

import geofasc.swing.model.DefaultFigureModel;
import geofasc.swing.model.FigureModel;
import geofasc.swing.plaf.BasicFigureUI;
import geofasc.swing.plaf.FigureUI;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;

import javax.swing.JPanel;
import javax.swing.UIManager;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * <code>AbstractFigure</code> serves as the abstract base class for all
 * components visualizing 2-dimensional geometric figures like rectangle, circle
 * or line segments.</p>
 * 
 * <code>AbstractFigure</code> and its sub-classes provide an object-oriented
 * and Swing-compliant approach to (geometric) figures: Actually, any figure can
 * be drawn by calling a <code>draw</code>-method on a {@link Graphics} object.
 * But figures drawn that way can not be treated and accessed as objects/
 * components and no distinction between them is possible in object-oriented
 * manner.</p>
 * 
 * Since an <code>AbstractFigure</code> is abstract it can not be seen as
 * specific figure. It rather represents the rectangular bounding box of a
 * specific figure and provides access for reading its size and location (in
 * fact it delegates the calls to {@link FigureModel}, see there for more
 * explanations). Of course, arranging figures is most meaningful on a common
 * parent component with a null-layout but is also possible with any other
 * layout even besides other Swing components. Moreover this class provides
 * access for the common visual properties of a figure like its line color.</p>
 * 
 * Note, that an <code>AbstractFigure</code>'s properties must not mapped
 * one-to-one to its corresponding Swing properties. That means that, for
 * example a rectangle figure, can have another size than its visualizing
 * component. This distinction is necessary to be compatible with the layout
 * managers that might resize components themselves.
 * 
 * @version 0.12 06/11/11
 * @author Christian Wolf
 * @see FigureModel
 * @see DefaultFigureModel
 * @see FigureUI
 * @see BasicFigureUI
 * 
 */
@SuppressWarnings("serial")
public abstract class AbstractFigure extends JPanel implements ChangeListener,
		ContainerListener {

	private static final String uiClassID = "FigureUI";

	private FigureModel mModel;
	private boolean mIsSelfContainerListener;

	/**
	 * Constructs a new <code>AbstractFigure</code> at <code>(x,y)</code>
	 * -location <code>(0,0)</code>.
	 */
	public AbstractFigure() {
		this(0, 0);
	}

	/**
	 * Constructs a new <code>AbstractFigure</code> at the given location.
	 * 
	 * @param x
	 *            the initial x-location of this figure
	 * @param y
	 *            the initial y-location of this figure
	 */
	public AbstractFigure(int x, int y) {
		super(null, true);
		initModel(x, y);
		setSelfContainerListener(true);
		updateUI();
	}

	/**
	 * {@inheritDoc} </p>
	 * 
	 * Overridden for automatically calling methods {@link #revalidate()} and
	 * {@link #repaint()} if a child component is added to this figure and the
	 * figure has been already displayed.
	 * 
	 * Actually this has to be done manually calling the mentioned methods for
	 * efficiency reasons (e.g., adding multiple components to this figure). To
	 * do more efficiently call {@link #setSelfContainerListener(boolean)} with
	 * <code>false</code> to deactivate this figure as its own container
	 * listener, then add the components and fo revalidating as well as
	 * repainting manually.
	 */
	@Override
	public void componentAdded(ContainerEvent e) {
		Component c = e.getChild();
		if (c == null)
			return;

		revalidate();

		// only repaint the dirty region of the component
		repaint(c.getBounds());
	}

	/**
	 * {@inheritDoc} </p>
	 * 
	 * Overridden for automatically calling methods {@link #revalidate()} and
	 * {@link #repaint()} if a child component is removed from this figure and
	 * the figure has been already displayed.
	 * 
	 * Actually this has to be done manually calling the mentioned methods for
	 * efficiency reasons (e.g., removing multiple or all child components of
	 * this figure). To do more efficiently call
	 * {@link #setSelfContainerListener(boolean)} with <code>false</code> to
	 * deactivate this figure as its own container listener, then remove the
	 * components and do validating as well as repainting manually.
	 */
	@Override
	public void componentRemoved(ContainerEvent e) {
		Component c = e.getChild();
		if (c == null)
			return;

		revalidate();

		// only repaint the dirty region of the component
		repaint(c.getBounds());
	}

	/**
	 * Creates a default model which is used after constructing this figure. The
	 * model can be changed after construction (see
	 * {@link #setModel(FigureModel)}).
	 * 
	 * @return the default model
	 */
	protected abstract FigureModel createDefaultModel();

	/** {@inheritDoc} */
	@Override
	public Dimension getMinimumSize() {
		if (getModel() != null && getModel().getBounds() != null) {
			Rectangle bounds = getModel().getBounds();

			return new Dimension(bounds.width + 1, bounds.height + 1);
		} else
			return new Dimension();
	}

	/**
	 * Returns the data model of this figure.
	 * 
	 * @return the current model
	 * @see FigureModel
	 */
	public FigureModel getModel() {
		return mModel;
	}

	/** {@inheritDoc} */
	@Override
	public Dimension getPreferredSize() {
		return getMinimumSize();
	}

	/** {@inheritDoc} */
	@Override
	public FigureUI getUI() {
		return (FigureUI) ui;
	}

	/** {@inheritDoc} */
	@Override
	public String getUIClassID() {
		return uiClassID;
	}

	private void initModel(int x, int y) {
		mModel = createDefaultModel();

		if (mModel != null) {
			mModel.addChangeListener(this);
			mModel.setLocation(x, y);
		}
	}

	/** {@inheritDoc} */
	@Override
	public void setMaximumSize(Dimension maximumSize) {
		Dimension minSize = getMinimumSize();
		if (maximumSize.width >= minSize.width
				&& maximumSize.height >= minSize.height)
			super.setMaximumSize(maximumSize);
	}

	/**
	 * Sets the model of this figure.
	 * 
	 * @param model
	 *            the new model
	 */
	public void setModel(FigureModel model) {
		if (model != null) {
			FigureModel oldModel = getModel();

			if (oldModel != null)
				oldModel.removeChangeListener(this);

			mModel = model;
			mModel.addChangeListener(this);
			firePropertyChange("model", oldModel, mModel);

			if (mModel != oldModel) {
				revalidate();
				repaint();
			}
		}
	}

	/**
	 * Sets this figure as its own container listener.
	 * 
	 * @param b
	 *            true or false
	 * @see #componentAdded(ContainerEvent)
	 * @see #componentRemoved(ContainerEvent)
	 */
	public void setSelfContainerListener(boolean b) {
		mIsSelfContainerListener = b;

		removeContainerListener(this);
		if (mIsSelfContainerListener)
			addContainerListener(this);
	}
	
	/**
	 * Sets the look and feel (L&F) object that renders this figure.
	 * 
	 * @param ui
	 *            the new L&F object
	 */
	public void setUI(FigureUI ui) {
		super.setUI(ui);
	}

	/** {@inheritDoc} */
	@Override
	public void stateChanged(ChangeEvent e) {
		Container parent = getParent();

		if (parent == null
				|| (parent != null && parent.getLayout() == null
						&& getModel() != null && getModel().getBounds() != null)) {

			Rectangle figureBounds = getModel().getBounds();
			setBounds(figureBounds.x, figureBounds.y, figureBounds.width + 1,
					figureBounds.height + 1);
		}

		revalidate();
		repaint();
	}

	/** {@inheritDoc} */
	@Override
	public void updateUI() {
		if (UIManager.get(getUIClassID()) != null) {
			setUI((FigureUI) UIManager.getUI(this));
		} else {
			setUI(new BasicFigureUI());
		}
	}

	// Convenience methods as syntactic sugar which just delegate
	// their calls to the model. Getter methods return the values
	// as returned by the model and might be null!

	/**
	 * See {@link FigureModel#getDirection()}.
	 */
	public double getFigureDirection() {
		if (getModel() != null)
			return getModel().getDirection();
		else
			return -1;
	}

	/**
	 * See {@link FigureModel#getBounds()}.
	 */
	public Rectangle getFigureBounds() {
		if (getModel() != null)
			return getModel().getBounds();
		else
			return null;
	}

	/**
	 * See {@link FigureModel#getBounds(Rectangle)}.
	 */
	public Rectangle getFigureBounds(Rectangle bounds) {
		if (getModel() != null)
			return getModel().getBounds(bounds);
		else
			return null;
	}

	/**
	 * See {@link FigureModel#getLineColor()}.
	 */
	public Color getFigureLineColor() {
		if (getModel() != null)
			return getModel().getLineColor();
		else
			return null;
	}

	/**
	 * See {@link FigureModel#getFillColor()}.
	 */
	public Color getFigureFillColor() {
		if (getModel() != null)
			return getModel().getFillColor();
		else
			return null;
	}

	/**
	 * See {@link FigureModel#getHeight()}.
	 */
	public int getFigureHeight() {
		if (getModel() != null)
			return getModel().getHeight();
		else
			return 0;
	}

	/**
	 * See {@link FigureModel#getLocation()}.
	 */
	public Point getFigureLocation() {
		if (getModel() != null)
			return getModel().getLocation();
		else
			return null;
	}

	/**
	 * See {@link FigureModel#getLocationX()}.
	 */
	public int getFigureLocationX() {
		if (getModel() != null)
			return getModel().getLocationX();
		else
			return 0;
	}

	/**
	 * See {@link FigureModel#getLocationY()}.
	 */
	public int getFigureLocationY() {
		if (getModel() != null)
			return getModel().getLocationY();
		else
			return 0;
	}

	/**
	 * See {@link FigureModel#getSize()}.
	 */
	public Dimension getFigureSize() {
		if (getModel() != null)
			return getModel().getSize();
		else
			return null;
	}

	/**
	 * See {@link FigureModel#getWidth()}.
	 */
	public int getFigureWidth() {
		if (getModel() != null)
			return getModel().getWidth();
		else
			return 0;
	}

	/**
	 * See {@link FigureModel#isFilled()}.
	 */
	public boolean isFigureFilled() {
		if (getModel() != null)
			return getModel().isFilled();
		else
			return false;
	}

	/**
	 * Returns whether this figure is its own container listener.
	 * 
	 * @return true or false
	 * @see #componentAdded(ContainerEvent)
	 * @see #componentRemoved(ContainerEvent)
	 * @see #setSelfContainerListener(boolean)
	 */
	public boolean isSelfContainerListener() {
		return mIsSelfContainerListener;
	}
	
	/**
	 * See {@link FigureModel#moveLocationBy(int)}.
	 */
	public void moveFigureLocationBy(int pixels) {
		if (getModel() != null)
			getModel().moveLocationBy(pixels);
	}

	/**
	 * See {@link FigureModel#moveLocationBy(int, int)}.
	 */
	public void moveFigureLocationBy(int dx, int dy) {
		if (getModel() != null)
			getModel().moveLocationBy(dx, dy);
	}

	/**
	 * See {@link FigureModel#moveLocationBy(Point)}.
	 */
	public void moveFigureLocationBy(Point dPoint) {
		if (getModel() != null)
			getModel().moveLocationBy(dPoint);
	}

	/**
	 * See {@link FigureModel#moveLocationXBy(int)}.
	 */
	public void moveFigureLocationXBy(int dx) {
		if (getModel() != null)
			getModel().moveLocationXBy(dx);
	}

	/**
	 * See {@link FigureModel#moveLocationYBy(int)}.
	 */
	public void moveFigureLocationYBy(int dy) {
		if (getModel() != null)
			getModel().moveLocationYBy(dy);
	}

	/**
	 * See {@link FigureModel#setDirection(double)}.
	 */
	public void setFigureDirection(double direction) {
		if (getModel() != null)
			getModel().setDirection(direction);
	}

	/**
	 * See {@link FigureModel#setFillColor(Color)}.
	 */
	public void setFigureFillColor(Color fillColor) {
		if (getModel() != null)
			getModel().setFillColor(fillColor);
	}

	/**
	 * See {@link FigureModel#isFilled()}.
	 */
	public void setFigureFilled(boolean isFilled) {
		if (getModel() != null)
			getModel().setFilled(isFilled);
	}

	/**
	 * See {@link FigureModel#setLineColor(Color)}.
	 */
	public void setFigureLineColor(Color lineColor) {
		if (getModel() != null)
			getModel().setLineColor(lineColor);
	}

	/**
	 * See {@link FigureModel#setLocation(int, int)}.
	 */
	public void setFigureLocation(int x, int y) {
		if (getModel() != null)
			getModel().setLocation(x, y);
	}

	/**
	 * See {@link FigureModel#setLocation(Point)}.
	 */
	public void setFigureLocation(Point p) {
		if (getModel() != null)
			getModel().setLocation(p);
	}

	/**
	 * See {@link FigureModel#setLocationX(int)}.
	 */
	public void setFigureLocationX(int x) {
		if (getModel() != null)
			getModel().setLocationX(x);
	}

	/**
	 * See {@link FigureModel#setLocationY(int)}.
	 */
	public void setFigureLocationY(int y) {
		if (getModel() != null)
			getModel().setLocationY(y);
	}

	/**
	 * See {@link FigureModel#turnDirectionTo(double)}.
	 */
	public void turnFigureDirectionTo(double direction) {
		if (getModel() != null)
			getModel().turnDirectionTo(direction);
	}

	/**
	 * See {@link FigureModel#turnDirectionBy(double)}.
	 */
	public void turnFigureDirectionBy(double dDirection) {
		if (getModel() != null)
			getModel().turnDirectionBy(dDirection);
	}

}
