/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'Polyline.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing;

import geofasc.swing.model.DefaultPolylineModel;
import geofasc.swing.model.FigureModel;
import geofasc.swing.model.PolylineModel;
import geofasc.swing.plaf.BasicPolylineUI;
import geofasc.swing.plaf.PolylineUI;

import java.awt.Point;

import javax.swing.UIManager;

/**
 * <code>Polyline</code> is the component used for visualizing a polyline or
 * polygon figure. The polyline is interpreted as a polygon if it is closed. See
 * {@link PolylineModel} for more information on how a (closed) polyline figure
 * is represented. This class is inherited from {@link AbstractFigure}.
 * 
 * @version 0.1 08/11/10
 * @author Christian Wolf
 * @see DefaultPolylineModel
 * @see PolylineUI
 * @see BasicPolylineUI
 * 
 */
@SuppressWarnings("serial")
public class Polyline extends AbstractFigure {

	private static final String uiClassID = "PolygonUI";

	private static Point[] createPoints(int[] xPoints, int[] yPoints) {
		if (xPoints == null || yPoints == null
				|| xPoints.length != yPoints.length)
			throw new IllegalArgumentException(
					"argument(s) xPoints[] or yPoints[] is null or they have different lengths!");

		Point[] points = new Point[xPoints.length];
		for (int i = 0; i < xPoints.length; i++)
			points[i] = new Point(xPoints[i], yPoints[i]);

		return points;
	}

	/**
	 * Constructs a new empty <code>Polyline</code> with none points.
	 */
	public Polyline() {
		super();
	}

	/**
	 * Constructs a new <code>Polyline</code> with the given points at the given
	 * location.
	 * 
	 * @param x
	 *            the initial x-location of this polyline
	 * @param y
	 *            the initial y-location of this polyline
	 * @param xPoints
	 *            the initial x-locations of all points of this polyline
	 * @param yPoints
	 *            the initial y-locations of all points of this polyline
	 */
	public Polyline(int x, int y, int[] xPoints, int[] yPoints) {
		this(x, y, createPoints(xPoints, yPoints));
	}

	/**
	 * Constructs a new <code>Polyline</code> with the given points at the given
	 * location.
	 * 
	 * @param x
	 *            the initial x-location of this polyline
	 * @param y
	 *            the initial y-location of this polyline
	 * @param points
	 *            the inital points of this polyline
	 */
	public Polyline(int x, int y, Point... points) {
		super(x, y);

		if (points == null)
			throw new NullPointerException("argument points[] is null!");

		getModel().setPoints(points);
	}

	/**
	 * Constructs a new <code>Polyline</code> with the given points at
	 * <code>(x,y)-</code> location <code>(0,0)</code>.
	 * 
	 * @param xPoints
	 *            the initial x-locations of all points of this polyline
	 * @param yPoints
	 *            the initial y-locations of all points of this polyline
	 */
	public Polyline(int[] xPoints, int[] yPoints) {
		this(0, 0, xPoints, yPoints);
	}

	/**
	 * Constructs a new <code>Polyline</code> with the given points at
	 * <code>(x,y)-</code> location <code>(0,0)</code>.
	 * 
	 * @param points
	 *            the inital points of this polyline
	 */
	public Polyline(Point... points) {
		this(0, 0, points);
	}

	/** {@inheritDoc} */
	@Override
	protected FigureModel createDefaultModel() {
		return new DefaultPolylineModel();
	}

	/** {@inheritDoc} */
	@Override
	public PolylineModel getModel() {
		return (PolylineModel) super.getModel();
	}

	/** {@inheritDoc} */
	@Override
	public PolylineUI getUI() {
		return (PolylineUI) ui;
	}

	/** {@inheritDoc} */
	@Override
	public String getUIClassID() {
		return uiClassID;
	}

	/**
	 * Sets the model of this polyline.
	 * 
	 * @param model
	 *            the new model
	 */
	public void setModel(PolylineModel model) {
		super.setModel(model);
	}

	public void setUI(PolylineUI ui) {
		super.setUI(ui);
	}

	/** {@inheritDoc} */
	@Override
	public void updateUI() {
		if (UIManager.get(getUIClassID()) != null) {
			setUI((PolylineUI) UIManager.getUI(this));
		} else {
			setUI(new BasicPolylineUI());
		}
	}

	// Convenience methods as syntactic sugar which just delegate
	// their calls to the model. Getter methods return the values
	// as returned by the model and might be null!

	/** See {@link PolylineModel#addPoint(int, int)} */
	public boolean addPoint(int x, int y) {
		if (getModel() != null) {
			return getModel().addPoint(x, y);
		} else
			return false;
	}

	/** See {@link PolylineModel#addPoint(int, int, int)} */
	public boolean addPoint(int index, int x, int y) {
		if (getModel() != null) {
			return getModel().addPoint(index, x, y);
		} else
			return false;
	}

	/** See {@link PolylineModel#addPoint(Point)} */
	public boolean addPoint(Point point) {
		if (getModel() != null)
			return getModel().addPoint(point);
		else
			return false;
	}

	/** See {@link PolylineModel#addPoint(int, Point)} */
	public boolean addPoint(int index, Point point) {
		if (getModel() != null)
			return getModel().addPoint(index, point);
		else
			return false;
	}

	/** See {@link PolylineModel#getIndexOfPoint(int, int)} */
	public int getIndexOfPoint(int x, int y) {
		if (getModel() != null)
			return getModel().getIndexOfPoint(x, y);
		else
			return -1;
	}

	/** See {@link PolylineModel#getIndexOfPoint(Point)} */
	public int getIndexOfPoint(Point point) {
		if (getModel() != null)
			return getModel().getIndexOfPoint(point);
		else
			return -1;
	}

	/** See {@link PolylineModel#getNumberOfPoints()} */
	public int getNumberOfPoints() {
		if (getModel() != null)
			return getModel().getNumberOfPoints();
		else
			return 0;
	}

	/** See {@link PolylineModel#getPoint(int)} */
	public Point getPoint(int index) {
		if (getModel() != null)
			return getModel().getPoint(index);
		else
			return null;
	}

	/** See {@link PolylineModel#getPoints()} */
	public Point[] getPoints() {
		if (getModel() != null)
			return getModel().getPoints();
		else
			return null;
	}

	/** See {@link PolylineModel#getXPoints()} */
	public int[] getXPoints() {
		if (getModel() != null)
			return getModel().getXPoints();
		else
			return null;
	}

	/** See {@link PolylineModel#getYPoints()} */
	public int[] getYPoints() {
		if (getModel() != null)
			return getModel().getYPoints();
		else
			return null;
	}

	/** See {@link PolylineModel#isClosed()} */
	public boolean isClosed() {
		if (getModel() != null)
			return getModel().isClosed();
		else
			return false;
	}

	/** See {@link PolylineModel#removePoint(int)} */
	public boolean removePoint(int index) {
		if (getModel() != null)
			return getModel().removePoint(index);
		else
			return false;
	}

	/** See {@link PolylineModel#removePoint(int, int)} */
	public boolean removePoint(int x, int y) {
		if (getModel() != null)
			return getModel().removePoint(x, y);
		else
			return false;
	}

	/** See {@link PolylineModel#removePoint(Point)} */
	public boolean removePoint(Point point) {
		if (getModel() != null)
			return getModel().removePoint(point);
		else
			return false;
	}

	/** See {@link PolylineModel#setPoint(int, int, int)} */
	public Point setPoint(int index, int x, int y) {
		if (getModel() != null)
			return getModel().setPoint(index, x, y);
		else
			return null;
	}

	/** See {@link PolylineModel#setPoint(int, Point)} */
	public Point setPoint(int index, Point point) {
		if (getModel() != null)
			return getModel().setPoint(index, point);
		else
			return null;
	}

	/** See {@link PolylineModel#setPoints(Point[])} */
	public boolean setPoints(Point[] points) {
		if (getModel() != null)
			return getModel().setPoints(points);
		else
			return false;
	}

	/** See {@link PolylineModel#toAWTPolygon()} */
	public java.awt.Polygon toAWTPolygon() {
		if (getModel() != null)
			return getModel().toAWTPolygon();
		else
			return null;
	}

}
