/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'Rectangle.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing;

import geofasc.swing.model.DefaultRectangleModel;
import geofasc.swing.model.FigureModel;
import geofasc.swing.model.FigureModelExtended;
import geofasc.swing.model.RectangleModel;
import geofasc.swing.plaf.BasicRectangleUI;
import geofasc.swing.plaf.RectangleUI;

import java.awt.Dimension;

import javax.swing.UIManager;

/**
 * <code>Rectangle</code> is the component used for visualizing a rectangle
 * figure. See {@link RectangleModel} for more information on how a rectangle
 * figure is represented. This class is inherited from {@link AbstractFigure}.
 * 
 * @version 0.1 08/11/10
 * @author Christian Wolf
 * @see DefaultRectangleModel
 * @see RectangleUI
 * @see BasicRectangleUI
 * 
 */
@SuppressWarnings("serial")
public class Rectangle extends AbstractFigure {

	private static final String uiClassID = "RectangleUI";

	/**
	 * Constructs a new empty <code>Rectangle</code> with width and height equal
	 * to 0.
	 */
	public Rectangle() {
		this(0, 0);
	}

	/**
	 * Constructs a new <code>Rectangle</code> with the given size.
	 * 
	 * @param width
	 *            the initial width of this rectangle
	 * @param height
	 *            the initial height of this rectangle
	 */
	public Rectangle(int width, int height) {
		this(0, 0, width, height);
	}

	/**
	 * Constructs a new <code>Rectangle</code> with the given bounds (size and
	 * location).
	 * 
	 * @param x
	 *            the initial x-location of this rectangle
	 * @param y
	 *            the initial y-location of this rectangle
	 * @param width
	 *            the initial width of this rectangle
	 * @param height
	 *            the initial height of this rectangle
	 */
	public Rectangle(int x, int y, int width, int height) {
		super(x, y);
		initModel(width, height);
	}

	/** {@inheritDoc} */
	@Override
	protected FigureModel createDefaultModel() {
		return new DefaultRectangleModel();
	}

	/** {@inheritDoc} */
	@Override
	public RectangleModel getModel() {
		return (RectangleModel) super.getModel();
	}

	/** {@inheritDoc} */
	@Override
	public RectangleUI getUI() {
		return (RectangleUI) ui;
	}

	/** {@inheritDoc} */
	@Override
	public String getUIClassID() {
		return uiClassID;
	}

	private void initModel(int width, int height) {
		if (getModel() != null) {
			RectangleModel model = getModel();
			model.setSize(width, height);
			model.setRoundedCorners(false);
			model.setArcHeight(5);
			model.setArcWidth(5);
		}
	}

	/**
	 * Sets the model of this rectangle.
	 * 
	 * @param model
	 *            the new model of this rectangle
	 */
	public void setModel(RectangleModel model) {
		super.setModel(model);
	}

	/**
	 * Sets the look and feel (L&F) object that renders this rectangle.
	 * 
	 * @param ui
	 *            the new L&F object
	 */
	public void setUI(RectangleUI ui) {
		super.setUI(ui);
	}

	/** {@inheritDoc} */
	@Override
	public void updateUI() {
		if (UIManager.get(getUIClassID()) != null) {
			setUI((RectangleUI) UIManager.getUI(this));
		} else {
			setUI(new BasicRectangleUI());
		}
	}

	// Convenience methods as syntactic sugar which just delegate
	// their calls to the model. Getter methods return the values
	// as returned by the model and might be null!

	/**
	 * See {@link RectangleModel#getArcHeight()}.
	 */
	public int getArcHeight() {
		if (getModel() != null)
			return getModel().getArcHeight();
		else
			return 0;
	}

	/**
	 * See {@link RectangleModel#getArcSize()}.
	 */
	public Dimension getArcSize() {
		if (getModel() != null)
			return getModel().getArcSize();
		else
			return null;
	}

	/**
	 * See {@link RectangleModel#getArcWidth()}.
	 */
	public int getArcWidth() {
		if (getModel() != null)
			return getModel().getArcWidth();
		else
			return 0;
	}

	/**
	 * See {@link RectangleModel#hasRoundedCorners()}.
	 */
	public boolean hasRoundedCorners() {
		if (getModel() != null)
			return getModel().hasRoundedCorners();
		else
			return false;
	}

	/**
	 * See {@link FigureModelExtended#moveHeightBy(int)}.
	 */
	public void moveFigureHeightBy(int dHeight) {
		if (getModel() != null)
			getModel().moveHeightBy(dHeight);
	}

	/**
	 * See {@link FigureModelExtended#moveSizeBy(Dimension)}.
	 */
	public void moveFigureSizeBy(Dimension dSize) {
		if (getModel() != null)
			getModel().moveSizeBy(dSize);
	}

	/**
	 * See {@link FigureModelExtended#moveSizeBy(int, int)}.
	 */
	public void moveFigureSizeBy(int dWidth, int dHeight) {
		if (getModel() != null)
			getModel().moveSizeBy(dWidth, dHeight);
	}

	/**
	 * See {@link FigureModelExtended#moveWidthBy(int)}.
	 */
	public void moveFigureWidthBy(int dWidth) {
		if (getModel() != null)
			getModel().moveWidthBy(dWidth);
	}

	/**
	 * See {@link FigureModelExtended#setBounds(int, int, int, int)}.
	 */
	public void setFigureBounds(int x, int y, int width, int height) {
		if (getModel() != null)
			getModel().setBounds(x, y, width, height);
	}

	/**
	 * See {@link FigureModelExtended#setBounds(Rectangle)}.
	 */
	public void setFigureBounds(java.awt.Rectangle bounds) {
		if (getModel() != null)
			getModel().setBounds(bounds);
	}

	/**
	 * See {@link FigureModelExtended#setHeight(int)}.
	 */
	public void setFigureHeight(int height) {
		if (getModel() != null)
			getModel().setHeight(height);
	}

	/**
	 * See {@link FigureModelExtended#setSize(Dimension)}.
	 */
	public void setFigureSize(Dimension size) {
		if (getModel() != null)
			getModel().setSize(size);
	}

	/**
	 * See {@link FigureModelExtended#setSize(int, int)}.
	 */
	public void setFigureSize(int width, int height) {
		if (getModel() != null)
			getModel().setSize(width, height);
	}

	/**
	 * See {@link FigureModelExtended#setWidth(int)}.
	 */
	public void setFigureWidth(int width) {
		if (getModel() != null)
			getModel().setWidth(width);
	}

	/**
	 * See {@link RectangleModel#setArcHeight(int)}.
	 */
	public void setArcHeight(int arcHeight) {
		if (getModel() != null)
			getModel().setArcHeight(arcHeight);
	}

	/**
	 * See {@link RectangleModel#setArcSize(Dimension)}.
	 */
	public void setArcSize(Dimension d) {
		if (getModel() != null)
			getModel().setArcSize(d);
	}

	/**
	 * See {@link RectangleModel#setArcSize(int, int)}.
	 */
	public void setArcSize(int arcWidth, int arcHeight) {
		if (getModel() != null)
			getModel().setArcSize(arcWidth, arcHeight);
	}

	/**
	 * See {@link RectangleModel#setArcWidth(int)}.
	 */
	public void setArcWidth(int arcWidth) {
		if (getModel() != null)
			getModel().setArcWidth(arcWidth);
	}

	/**
	 * See {@link RectangleModel#setRoundedCorners(boolean)}.
	 */
	public void setRoundedCorners(boolean roundRect) {
		if (getModel() != null)
			getModel().setRoundedCorners(roundRect);
	}

}
