/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'Pencil.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package geofasc.swing.demo;

import geofasc.swing.Polyline;
import geofasc.swing.Rectangle;

import java.awt.Color;
import java.awt.Point;

/**
 * Ein <code>Windrad</code> besteht aus einem Mast und einem {@link TwinFluegel}
 * . Die Rotation des Windrades kann gestartet und auch wieder gestoppt werden.
 * 
 * @author Christian Wolf
 */
@SuppressWarnings("serial")
public class Windrad extends Rectangle {

	// Attribute

	/*
	 * Twin-Flügel des Windrades
	 */
	private TwinFluegel fluegel;

	/*
	 * Mast des Windrades
	 */
	private Polyline mast;
	
	/*
	 * Boolesche Variable, die die Anforderung zum Stoppen des sich drehenden
	 * Windrades speichern soll.
	 */
	private boolean stoppe;

	// Konstruktoren

	/**
	 * Erzeugt ein neues Windrad-Objekt.
	 */
	public Windrad() {
		this(0);
	}

	/**
	 * Erzeugt ein neues Windrad-Objekt mit dem übergebenen Winkel als
	 * Drehwinkel seiner Flügel.
	 */
	public Windrad(double winkel) {
		this(0, 0, winkel);
	}

	/**
	 * Erzeugt ein neues Windrad-Objekt an der übergebenen Position und mit dem
	 * übergebenen Winkel als Drehwinkel seiner Flügel.
	 */
	public Windrad(int x, int y, double winkel) {
		stoppe = false;
		baueWindrad(winkel);
		
		setFigureLocation(x, y);
		setFigureSize(150, 250);
		setFigureLineColor(Color.LIGHT_GRAY);
	}

	// Methoden

	/**
	 * Baut das Windrad aus Mast und Flügeln zusammen.
	 */
	private void baueWindrad(double winkel) {
		fluegel = new TwinFluegel(winkel);
		
		mast = new Polyline(new Point(73, 75), new Point(63,
				248), new Point(93, 248), new Point(78, 75), new Point(73, 75));
		mast.setFigureFilled(true);
		mast.setFigureFillColor(Color.WHITE);

		add(fluegel);
		add(mast);
	}

	/**
	 * Beschleunigt die Flügel des Windrades bevor sie sich mit ungefähr
	 * konstanter Geschwindigkeit drehen.
	 */
	private void beschleunigeFluegel() {

		int verzoegerung = 530;

		// 30 Beschleunigungen
		for (int i = 1; i <= 30; i++) {
			fluegel.dreheUm(3);
			verzoegere(verzoegerung);
			verzoegerung = verzoegerung - 16;
		}
	}

	/**
	 * Dreht die Flügel des Windrades mit konstanter Geschwindigkeit.
	 */
	private void dreheFluegel() {
		
		for (int i=0; i<72; i=i+1) {
			fluegel.dreheUm(5);
			verzoegere(50);
		}
		
		int i=0;
		while(i<72) {
			fluegel.dreheUm(5);
			verzoegere(50);
			i = i+1;
		}
	}

	/**
	 * Entschleunigt die Flügel des Windrades und bremst diese bis zum
	 * Stillstand ab, nachdem sie sich mit ungefähr konstanter Geschwindigkeit
	 * gedreht haben.
	 */
	private void entschleunigeFluegel() {
		int verzoegerung = 30;
		for (int i = 1; i <= 30; i++) {
			fluegel.dreheUm(5);
			verzoegere(verzoegerung);
			verzoegerung += 10;
		}
	}

	/**
	 * Lässt den Rotor des Windrades mit seinen Flügeln rotieren. Zunächst
	 * werden die Flügel beschleunigt, dann drehen diese für bestimmte Zeit und
	 * werden letztlich wieder entschleunigt.
	 */
	public void starteRotieren() {
		Thread drehThread = new Thread() {

			@Override
			public void run() {
				stoppe = false;

				beschleunigeFluegel();
				dreheFluegel();
				entschleunigeFluegel();

				interrupt();
			}

		};

		drehThread.start();
	}

	/**
	 * Stoppt den Rotor des Windrades. Vielmehr wird das Stoppen veranlasst,
	 * wozu intern eine boolesche Variable auf 'false' gesetzt wird, die an
	 * anderer Stelle ausgewertet werden muss.
	 */
	public void stoppeRotieren() {
		stoppe = true;
	}

	/**
	 * Verzögert das Ausführen des Programms um die übergebene Anzahl von
	 * Millisekunden.
	 */
	public static void verzoegere(int millis) {
		try {
			Thread.sleep(millis);
		} catch (Exception e) {
		}
	}

}
