/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'DefaultFigureModel.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.model;


import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.EventListenerList;

/**
 * <code>DefaultFigureModel</code> is the default implementation of
 * {@link FigureModel}.
 * 
 * @version 0.1 11/08/10
 * @author Christian Wolf
 * 
 */
public abstract class DefaultFigureModel implements FigureModel {

	private double mDirection;
	protected double mLocationX;
	protected double mLocationY;
	private Color mLineColor;
	private Color mFillColor;
	private boolean mIsFilled;
	private EventListenerList mListenerList = new EventListenerList();

	protected transient ChangeEvent mChangeEvent = null;

	/** {@inheritDoc} */
	@Override
	public void addChangeListener(ChangeListener l) {
		mListenerList.add(ChangeListener.class, l);
	}

	/**
	 * Determines the new x-location of this model's figure dependent
	 * on the direction.
	 * 
	 * @param pixels the length in pixels to move by
	 * @return the new x-location
	 */
	private double determineX(int pixels) {
		double d = -mDirection * Math.PI / 180;
  		return mLocationX + pixels * Math.cos(d);
	}
	
	/**
	 * Determines the new y-location of this model's figure dependent
	 * on the direction.
	 * 
	 * @param pixels the length in pixels to move by
	 * @return the new y-location
	 */
	private double determineY(int pixels) {
		double d = -mDirection * Math.PI / 180;
  		return mLocationY + pixels * Math.sin(d);
	}
	
	/**
	 * Notifies the listeners of this model.
	 */
	protected void fireStateChanged() {
		// Guaranteed to return a non-null array
		Object[] listeners = mListenerList.getListenerList();
		// Process the listeners last to first, notifying
		// those that are interested in this event
		for (int i = listeners.length - 2; i >= 0; i -= 2) {
			if (listeners[i] == ChangeListener.class) {
				// Lazily create the event:
				if (mChangeEvent == null)
					mChangeEvent = new ChangeEvent(this);
				((ChangeListener) listeners[i + 1]).stateChanged(mChangeEvent);
			}
		}
	}

	/** {@inheritDoc} */
	@Override
	public abstract Rectangle getBounds();

	/** {@inheritDoc} */
	@Override
	public Rectangle getBounds(Rectangle bounds) {
		if (bounds == null)
			return getBounds();
		else {
			bounds.setBounds(round(mLocationX), round(mLocationY), getWidth(), getHeight());
			return bounds;
		}
	}

	/** {@inheritDoc} */
	@Override
	public double getDirection() {
		return mDirection;
	}
	
	/** {@inheritDoc} */
	@Override
	public Color getFillColor() {
		return mFillColor;
	}

	/** {@inheritDoc} */
	@Override
	public int getHeight() {
		if (getBounds() != null)
			return getBounds().height;
		else
			return 0;
	}

	/** {@inheritDoc} */
	@Override
	public Color getLineColor() {
		return mLineColor;
	}

	/** {@inheritDoc} */
	@Override
	public Point getLocation() {
		return new Point(round(mLocationX), round(mLocationY));
	}

	/** {@inheritDoc} */
	@Override
	public int getLocationX() {
		return round(mLocationX);
	}

	/** {@inheritDoc} */
	@Override
	public int getLocationY() {
		return round(mLocationY);
	}

	/** {@inheritDoc} */
	@Override
	public Dimension getSize() {
		if (getBounds() != null)
			return getBounds().getSize();
		else
			return new Dimension();
	}

	/** {@inheritDoc} */
	@Override
	public int getWidth() {
		if (getBounds() != null)
			return getBounds().width;
		else
			return 0;
	}

	/** {@inheritDoc} */
	@Override
	public boolean isFilled() {
		return mIsFilled;
	}

	/** {@inheritDoc} */
	@Override
	public void moveLocationBy(int pixels) {
		if (pixels == 0)
			return;
		
		setLocationAsDouble(determineX(pixels), determineY(pixels));
	}
	
	/** {@inheritDoc} */
	@Override
	public void moveLocationBy(int dx, int dy) {
		setLocationAsDouble(mLocationX + dx, mLocationY + dy);
	}

	/** {@inheritDoc} */
	@Override
	public void moveLocationBy(Point dPoint) {
		if (dPoint != null)
			setLocationAsDouble(mLocationX + dPoint.x, mLocationY + dPoint.y);
	}

	/** {@inheritDoc} */
	@Override
	public void moveLocationXBy(int dx) {
		setLocationAsDouble(mLocationX + dx, mLocationY);
	}

	/** {@inheritDoc} */
	@Override
	public void moveLocationYBy(int dy) {
		setLocationAsDouble(mLocationX, mLocationY + dy);
	}

	/** {@inheritDoc} */
	@Override
	public void removeChangeListener(ChangeListener l) {
		mListenerList.remove(ChangeListener.class, l);
	}

	/**
	 * See {@link Math#round(double)}.
	 */
	protected int round(double d) {
		return (int) Math.round(d);
	}
	
	/** {@inheritDoc} */
	@Override
	public void setDirection(double direction) {		
		if (mDirection == direction)
			return;
		
		mDirection = direction;
  		while (mDirection < 0) 
  			mDirection += 360;
  		while (mDirection >= 720)
  			mDirection -= 360;
  		
  		fireStateChanged();
	}
	
	/** {@inheritDoc} */
	@Override
	public void setFillColor(Color fillColor) {
		if (fillColor == null || fillColor.equals(mFillColor))
			return;

		mFillColor = fillColor;
		fireStateChanged();
	}

	/** {@inheritDoc} */
	@Override
	public void setFilled(boolean isFilled) {
		if (isFilled == mIsFilled)
			return;

		mIsFilled = isFilled;
		fireStateChanged();
	}

	/** {@inheritDoc} */
	@Override
	public void setLineColor(Color lineColor) {
		if (lineColor == null || lineColor.equals(mLineColor))
			return;

		mLineColor = lineColor;
		fireStateChanged();
	}

	/** {@inheritDoc} */
	@Override
	public void setLocation(int x, int y) {
		if (x == mLocationX && y == mLocationY)
			return;

		mLocationX = x;
		mLocationY = y;
		fireStateChanged();
	}

	// Internally, allows the more precise positioning of a figure
	// which is needed for the direction functionality
	private void setLocationAsDouble(double x, double y) {
		if (x == mLocationX && y == mLocationY)
			return;

		mLocationX = x;
		mLocationY = y;
		fireStateChanged();
	}
	
	/** {@inheritDoc} */
	@Override
	public void setLocation(Point p) {
		if (p != null)
			setLocation(p.x, p.y);
	}

	/** {@inheritDoc} */
	@Override
	public void setLocationX(int x) {
		if (x == mLocationX)
			return;

		mLocationX = x;
		fireStateChanged();
	}

	/** {@inheritDoc} */
	@Override
	public void setLocationY(int y) {
		if (y == mLocationY)
			return;

		mLocationY = y;
		fireStateChanged();
	}
	
	/** {@inheritDoc} */
	@Override
	public void turnDirectionBy(double dDirection) {
		setDirection(getDirection() + dDirection);
	}

	/** {@inheritDoc} */
	@Override
	public void turnDirectionTo(double direction) {
		setDirection(direction);
	}

}
