/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'FigureModel.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.model;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;

import javax.swing.event.ChangeListener;

/**
 * <code>FigureModel</code> is the interface for the data model of a
 * 2-dimensional figure primarily maintaining information about its size and its
 * location. </p>
 * 
 * In this model the size and location of a figure is interpreted as the size
 * and location of its <em>bounding box</em> (also called <em>bounds</em>). The
 * bounding box of a figure is the rectangle of minimum area size that encloses
 * the whole figure. At least its area is needed by a visualizing component to
 * show the figure entirely. In turn the location of the bounding box is given
 * by the point (with <code>x-</code> and <code>y-</code> coordinates) of its
 * top-left corner with respect to the 2-dimensional coordinate system of the
 * figure's parent component. Its size is given by its <code>width</code> and
 * <code>height</code></p>
 * 
 * Besides this basic geometric information, the model also provides a few
 * common visual properties, e.g., the figure's line color or fill color. Access
 * for all the mentioned information is provided by appropriate methods (and
 * convenience methods allowing also partial access) except methods for changing
 * the size. This makes sense since the bounding box is defined by the figure
 * and not vice versa. Otherwise changing the bounding box would have influence
 * on changing the figure itself.
 * 
 * @version 0.11 30/03/11
 * @author Christian Wolf
 * 
 */
public interface FigureModel {

	/**
	 * Adds a listener to this model which is notified on model changes.
	 * 
	 * @param listener
	 *            the listener to be added to this model
	 */
	void addChangeListener(ChangeListener listener);

	/**
	 * Gets the bounds (size and location) of this model's figure.
	 * 
	 * @return the current bounds
	 * @see #getBounds(Rectangle)
	 * @see #getLocation()
	 * @see #getSize()
	 */
	Rectangle getBounds();

	/**
	 * Gets the bounds (size and location) of this model's figure.
	 * 
	 * @param bounds
	 *            if not null this object will be overwritten with the bounds of
	 *            the figure and returned, otherwise a new object is returned
	 * 
	 * @return the current bounds
	 * @see #getBounds()
	 * @see #getLocation()
	 * @see #getSize()
	 */
	Rectangle getBounds(Rectangle bounds);

	/**
	 * Gets the direction (in degrees) of this model's figure.
	 * 
	 * @return the current direction
	 */
	double getDirection();

	/**
	 * Gets the fill color of this model's figure used when drawing it (has to
	 * be set extra).
	 * 
	 * @return the current fill color
	 * @see #setFilled(boolean)
	 */
	Color getFillColor();

	/**
	 * Gets the height of this model's figure
	 * 
	 * @return the current height
	 * @see #getWidth()
	 * @see #getSize()
	 */
	int getHeight();

	/**
	 * Gets the outer line color of this model's figure used when drawing it.
	 * 
	 * @return the current line color
	 */
	Color getLineColor();

	/**
	 * Gets the location as point of this model's figure.
	 * 
	 * @return the current location
	 * @see #getLocationX()
	 * @see #getLocationY()
	 */
	Point getLocation();

	/**
	 * Gets the x-location of this model's figure.
	 * 
	 * @return the current x-location
	 * @see #getLocationY()
	 * @see #getLocation()
	 */
	int getLocationX();

	/**
	 * Gets the y-location of this model's figure.
	 * 
	 * @return the current y-location
	 * @see #getLocationX()
	 * @see #getLocation()
	 */
	int getLocationY();

	/**
	 * Gets the size (width and height) of this model's figure.
	 * 
	 * @return the current size
	 * @see #getWidth()
	 * @see #getHeight()
	 */
	Dimension getSize();

	/**
	 * Gets the width of this model's figure.
	 * 
	 * @return the current width
	 * @see #getHeight()
	 * @see #getSize()
	 */
	int getWidth();

	/**
	 * Determines whether this model's figure is filled when it is drawn.
	 * 
	 * @return true or false
	 * @see #setFilled(boolean)
	 * @see #setFillColor(Color)
	 */
	boolean isFilled();

	/**
	 * Moves the location of this model's figure by the given length (as pixels)
	 * dependent on the direction of movement.
	 * 
	 * @param pixels
	 *            the pixels to move by
	 * @see #setDirection(double)
	 * @see #turnDirectionBy(double)
	 * @see #turnDirectionTo(double)
	 */
	void moveLocationBy(int pixels);

	/**
	 * Moves the location of this model's figure in <code>x-</code> and
	 * <code>y-</code> direction by <code>dx</code> and <code>dy</code>
	 * respectively.
	 * 
	 * @param dx
	 *            the <code>delta-x</code> to move the <code>x-</code>location
	 *            by
	 * @param dy
	 *            the <code>delta-y</code> to move the <code>y-</code>location
	 *            by
	 * @see #moveLocationBy(Point)
	 * @see #moveLocationXBy(int)
	 * @see #moveLocationYBy(int)
	 */
	void moveLocationBy(int dx, int dy);

	/**
	 * Moves the location of this model's figure in <code>x-</code> and
	 * <code>y-</code> direction as given by <code>dPoint</code>.
	 * 
	 * @param dPoint
	 *            the <code>delta-point</code> to move the location by
	 * @see #moveLocationBy(int, int)
	 * @see #moveLocationXBy(int)
	 * @see #moveLocationYBy(int)
	 */
	void moveLocationBy(Point dPoint);

	/**
	 * Moves the location of this model's figure in <code>x-</code>direction by
	 * <code>dx</code>.
	 * 
	 * @param dx
	 *            the <code>delta-x</code> to move the <code>x-</code>location
	 *            by
	 * @see #moveLocationYBy(int)
	 * @see #moveLocationBy(int, int)
	 * @see #moveLocationBy(Point)
	 */
	void moveLocationXBy(int dx);

	/**
	 * Moves the location of this model's figure in <code>y-</code>direction by
	 * <code>dy</code>.
	 * 
	 * @param dy
	 *            the <code>delta-y</code> to move the <code>y-</code>location
	 *            by
	 * @see #moveLocationXBy(int)
	 * @see #moveLocationBy(int, int)
	 * @see #moveLocationBy(Point)
	 */
	void moveLocationYBy(int dy);

	/**
	 * Removes a previously added listener from the model.
	 * 
	 * @param listener
	 *            the listener to be removed
	 * @see #addChangeListener(ChangeListener)
	 */
	void removeChangeListener(ChangeListener listener);

	/**
	 * Sets the direction (in degrees) of this model's figure.
	 * 
	 * @param direction
	 *            the new direction
	 */
	void setDirection(double direction);

	/**
	 * Sets the fill color of this model's figure.
	 * 
	 * @param fillColor
	 *            the new fill color
	 * @see #setFilled(boolean)
	 */
	void setFillColor(Color fillColor);

	/**
	 * Sets whether this model's figure is filled when it is drawn.
	 * 
	 * @param isFilled
	 *            true or false
	 * @see #setFillColor(Color)
	 */
	void setFilled(boolean isFilled);

	/**
	 * Sets the outer line color of this model's figure used when drawing it.
	 * 
	 * @param lineColor
	 *            the new line color
	 */
	void setLineColor(Color lineColor);

	/**
	 * Sets the location of this model's figure.
	 * 
	 * @param x
	 *            the new x-location
	 * @param y
	 *            the new y-location
	 * @see #setLocation(Point)
	 * @see #setLocationX(int)
	 * @see #setLocationY(int)
	 */
	void setLocation(int x, int y);

	/**
	 * Sets the location of this model's figure.
	 * 
	 * @param p
	 *            the new location
	 * @see #setLocation(int, int)
	 * @see #setLocationX(int)
	 * @see #setLocationY(int)
	 */
	void setLocation(Point p);

	/**
	 * Sets the x-location of this model's figure.
	 * 
	 * @param x
	 *            the new x-location
	 * @see #setLocationY(int)
	 * @see #setLocation(Point)
	 * @see #setLocation(int, int)
	 */
	void setLocationX(int x);

	/**
	 * Sets the y-location of this model's figure.
	 * 
	 * @param y
	 *            the new y-location
	 * @see #setLocationX(int)
	 * @see #setLocation(Point)
	 * @see #setLocation(int, int)
	 */
	void setLocationY(int y);

	/**
	 * Turns the current direction (in degrees) of this model's figure by the
	 * given delta direction.
	 * 
	 * @param dDirection
	 *            the <code>delta-direction</code> to turn the current direction
	 *            by
	 */
	void turnDirectionBy(double dDirection);

	/**
	 * Turns the direction (in degrees) of this model's figure to the given
	 * direction.
	 * 
	 * @param direction
	 *            the new direction to turn to
	 */
	void turnDirectionTo(double direction);

}
