/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'Canvas.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.tool;

import geofasc.swing.AbstractFigure;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;
import java.util.ArrayList;

import javax.swing.JLayeredPane;

/**
 * A <code>Canvas</code> is intended to be used as a container with a
 * null-layout for {@link AbstractFigure}s but also other {@link Component}s
 * (also besides figures).
 * 
 * @author Christian Wolf
 * @version 0.1 28/08/10
 * 
 */
@SuppressWarnings("serial")
public class Canvas extends JLayeredPane implements ContainerListener {

	private boolean mIsSelfContainerListener;
	
	/**
	 * Creates a new empty <code>Canvas</code>.
	 */
	public Canvas() {
		super();
		setSelfContainerListener(true);
	}

	/**
	 * Creates a new <code>Canvas</code> putting the passed components into it.
	 * 
	 * @param comps
	 *            the inital components to put into this canvas
	 */
	public Canvas(Component... comps) {
		super();
		add(comps);
		setSelfContainerListener(true);
	}
	
	/**
	 * {@inheritDoc} </p>
	 * 
	 * Overridden for automatically calling methods {@link #revalidate()} and
	 * {@link #repaint()} if a child component is added to this figure and the
	 * figure has been already displayed.
	 * 
	 * Actually this has to be done manually calling the mentioned methods for
	 * efficiency reasons (e.g., adding multiple components to this figure). To
	 * do more efficiently call {@link #setSelfContainerListener(boolean)} with
	 * <code>false</code> to deactivate this figure as its own container
	 * listener, then add the components and fo revalidating as well as
	 * repainting manually.
	 */
	@Override
	public void componentAdded(ContainerEvent e) {
		Component c = e.getChild();
		if (c == null)
			return;

		revalidate();

		// only repaint the dirty region of the component
		repaint(c.getBounds());
	}

	/**
	 * {@inheritDoc} </p>
	 * 
	 * Overridden for automatically calling methods {@link #revalidate()} and
	 * {@link #repaint()} if a child component is removed from this figure and
	 * the figure has been already displayed.
	 * 
	 * Actually this has to be done manually calling the mentioned methods for
	 * efficiency reasons (e.g., removing multiple or all child components of
	 * this figure). To do more efficiently call
	 * {@link #setSelfContainerListener(boolean)} with <code>false</code> to
	 * deactivate this figure as its own container listener, then remove the
	 * components and do validating as well as repainting manually.
	 */
	@Override
	public void componentRemoved(ContainerEvent e) {
		Component c = e.getChild();
		if (c == null)
			return;

		revalidate();

		// only repaint the dirty region of the component
		repaint(c.getBounds());
	}
	
	/** {@inheritDoc} */
	@Override
	public Component add(Component comp) {
		if (comp != null) {		
			super.add(comp);
			return comp;
		}
		else
			return null;
	}

	/**
	 * Adds the given components to this canvas.
	 * 
	 * @param comps the components to add
	 */
	public void add(Component... comps) {
		if (comps != null) {
			for (Component c: comps) {
				if (c != null)		
					super.add(c);
			}
		}
	}
	
	/** {@inheritDoc} */
	@Override
	public int getComponentCount() {
		return super.getComponentCount();
	}

	/** {@inheritDoc} */
	@Override
	public Component[] getComponents() {
		return super.getComponents();
	}

	/**
	 * Gets the number of figures in this canvas.
	 * 
	 * @return the number of figures
	 */
	public int getFigureCount() {
		AbstractFigure[] figures = getFigures();

		if (figures != null)
			return figures.length;
		else
			return 0;
	}

	/**
	 * Gets all the figures in this canvas.
	 * 
	 * @return all the figures or null
	 */
	public AbstractFigure[] getFigures() {
		Component[] comps = getComponents();

		if (comps == null || comps.length <= 0)
			return null;

		ArrayList<AbstractFigure> figures = new ArrayList<AbstractFigure>(
				comps.length);
		for (Component c : comps) {
			if (c instanceof AbstractFigure)
				figures.add((AbstractFigure) c);
		}

		return figures.toArray(new AbstractFigure[0]);
	}

	/** {@inheritDoc} */
	@Override
	public int getHeight() {
		return super.getHeight();
	}

	/** {@inheritDoc} */
	@Override
	public Point getLocation() {
		return super.getLocation();
	}
	
	/** {@inheritDoc} */
	@Override
	public Dimension getSize() {
		return super.getSize();
	}

	/** {@inheritDoc} */
	@Override
	public int getWidth() {
		return super.getWidth();
	}

	/** {@inheritDoc} */
	@Override
	public int getX() {
		return super.getX();
	}

	/** {@inheritDoc} */
	@Override
	public int getY() {
		return super.getY();
	}

	/** {@inheritDoc} */
	@Override
	public boolean isEnabled() {
		return super.isEnabled();
	}

	/**
	 * Returns whether this canvas is its own container listener.
	 * 
	 * @return true or false
	 * @see #componentAdded(ContainerEvent)
	 * @see #componentRemoved(ContainerEvent)
	 * @see #setSelfContainerListener(boolean)
	 */
	public boolean isSelfContainerListener() {
		return mIsSelfContainerListener;
	}
	
	/** {@inheritDoc} */
	@Override
	public boolean isVisible() {
		return super.isVisible();
	}

	/** {@inheritDoc} */
	@Override
	public void moveToBack(Component c) {
		super.moveToBack(c);
	}

	/** {@inheritDoc} */
	@Override
	public void moveToFront(Component c) {
		super.moveToFront(c);
	}

	/** {@inheritDoc} */
	@Override
	public void remove(Component c) {
		super.remove(c);
	}

	/**
	 * Removes the given components from this canvas.
	 * 
	 * @param comps the components to remove
	 */
	public void remove(Component... comps) {
		if (comps != null) {
			for (Component c: comps) {
				remove(c);
			}
		}
	}
	
	/** {@inheritDoc} */
	@Override
	public void removeAll() {
		super.removeAll();
	}

	/**
	 * Sets this canvas as its own container listener.
	 * 
	 * @param b
	 *            true or false
	 * @see #componentAdded(ContainerEvent)
	 * @see #componentRemoved(ContainerEvent)
	 */
	public void setSelfContainerListener(boolean b) {
		mIsSelfContainerListener = b;

		removeContainerListener(this);
		if (mIsSelfContainerListener)
			addContainerListener(this);
	}
	
	/** {@inheritDoc} */
	@Override
	public void setEnabled(boolean enabled) {
		super.setEnabled(enabled);
	}

	/** {@inheritDoc} */
	@Override
	public void setVisible(boolean visible) {
		super.setVisible(visible);
	}

}
