/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'LineSegment.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing;

import geofasc.swing.model.DefaultLineSegmentModel;
import geofasc.swing.model.FigureModel;
import geofasc.swing.model.LineSegmentModel;
import geofasc.swing.plaf.BasicLineSegmentUI;
import geofasc.swing.plaf.LineSegmentUI;

import java.awt.Point;

import javax.swing.UIManager;

/**
 * <code>LineSegment</code> is the component used for visualizing a line segment
 * figure. See {@link LineSegmentModel} for more information on how a line
 * segment figure is represented. This class is inherited from
 * {@link AbstractFigure}.
 * 
 * @version 0.1 08/11/10
 * @author Christian Wolf
 * @see DefaultLineSegmentModel
 * @see LineSegmentUI
 * @see BasicLineSegmentUI
 * 
 */
@SuppressWarnings("serial")
public class LineSegment extends AbstractFigure {

	private static final String uiClassID = "LineUI";

	/**
	 * Constructs a new empty <code>LineSegment</code> with length 0 and at
	 * <code>(x,y)-</code> location <code>(0,0)</code>.
	 */
	public LineSegment() {
		this(0, 0, 0, 0);
	}

	/**
	 * Constructs a new <code>LineSegment</code> with the given restricting
	 * points and at <code>(x,y)-</code> location <code>(0,0)</code>. The
	 * coordinates of the points refer to this components coordinate system. The
	 * location refers to this component's parent coordinate system.
	 * 
	 * @param x0
	 *            the initial x-location of the first point of this line segment
	 * @param y0
	 *            the initial y-location of the first point of this line segment
	 * @param x1
	 *            the initial x-location of the second point of this line
	 *            segment
	 * @param y1
	 *            the initial y-location of the second point of this line
	 *            segment
	 */
	public LineSegment(int x0, int y0, int x1, int y1) {
		this(new Point(x0, y0), new Point(x1, y1));
	}

	/**
	 * Constructs a new <code>LineSegment</code> with the given restricting
	 * points and at <code>(x,y)-</code> location <code>(0,0)</code>. The
	 * coordinates of the points refer to this components coordinate system. The
	 * location refers to this component's parent coordinate system.
	 * 
	 * @param firstPoint
	 *            the first point of this line segment
	 * @param secondPoint
	 *            the second point of this line segment
	 */
	public LineSegment(Point firstPoint, Point secondPoint) {
		this(0, 0, firstPoint, secondPoint);
	}

	/**
	 * Constructs a new <code>LineSegment</code> with the given restricting
	 * points and at the given location. The coordinates of the points refer to
	 * this components coordinate system. The location refers to this
	 * component's parent coordinate system.
	 * 
	 * @param x
	 *            the x-location of this line segment
	 * @param y
	 *            the y-location of this line segment
	 * @param x0
	 *            the initial x-location of the first point of this line segment
	 * @param y0
	 *            the initial y-location of the first point of this line segment
	 * @param x1
	 *            the initial x-location of the second point of this line
	 *            segment
	 * @param y1
	 *            the initial y-location of the second point of this line
	 *            segment
	 */
	public LineSegment(int x, int y, int x0, int y0, int x1, int y1) {
		this(0, 0, new Point(x0, y0), new Point(x1, y1));
	}

	/**
	 * Constructs a new <code>LineSegment</code> with the given restricting
	 * points and at the given location. The coordinates of the points refer to
	 * this components coordinate system. The location refers to this
	 * component's parent coordinate system.
	 * 
	 * @param x
	 *            the x-location of this line segment
	 * @param y
	 *            the y-location of this line segment
	 * @param firstPoint
	 *            the first point of the line segment
	 * @param secondPoint
	 *            the second point of the line segment
	 */
	public LineSegment(int x, int y, Point firstPoint, Point secondPoint) {
		super(x, y);

		if (firstPoint == null || secondPoint == null)
			throw new IllegalArgumentException(
					"argument(s) firstPoint or seconPoint is null");

		initModel(firstPoint, secondPoint);
	}

	private void initModel(Point firstPoint, Point secondPoint) {
		if (getModel() != null) {
			LineSegmentModel model = getModel();
			model.setFirstPoint(firstPoint);
			model.setSecondPoint(secondPoint);
		}
	}

	/** {@inheritDoc} */
	@Override
	public LineSegmentModel getModel() {
		return (LineSegmentModel) super.getModel();
	}

	/**
	 * Sets the model of this line segment.
	 * 
	 * @param model
	 *            the new model of this line segment
	 */
	public void setModel(LineSegmentModel model) {
		super.setModel(model);
	}

	/** {@inheritDoc} */
	@Override
	protected FigureModel createDefaultModel() {
		return new DefaultLineSegmentModel();
	}

	/**
	 * Sets the look and feel (L&F) object that renders this line segment.
	 * 
	 * @param ui
	 *            the new L&F object
	 */
	public void setUI(LineSegmentUI ui) {
		super.setUI(ui);
	}

	/** {@inheritDoc} */
	@Override
	public void updateUI() {
		if (UIManager.get(getUIClassID()) != null) {
			setUI((LineSegmentUI) UIManager.getUI(this));
		} else {
			setUI(new BasicLineSegmentUI());
		}
	}

	/** {@inheritDoc} */
	@Override
	public LineSegmentUI getUI() {
		return (LineSegmentUI) ui;
	}

	/** {@inheritDoc} */
	@Override
	public String getUIClassID() {
		return uiClassID;
	}

	// Convenience methods as syntactic sugar which just delegate
	// their calls to the model. Getter methods return the values
	// as returned by the model and might be null!

	/** See {@link LineSegmentModel#getFirstPoint()}. */
	public Point getFirstPoint() {
		if (getModel() != null)
			return getModel().getFirstPoint();
		else
			return null;
	}

	/** See {@link LineSegmentModel#getFirstPointX()}. */
	public int getFirstPointX() {
		if (getModel() != null)
			return getModel().getFirstPointX();
		else
			return 0;
	}
	
	/** See {@link LineSegmentModel#getFirstPointY()}. */
	public int getFirstPointY() {
		if (getModel() != null)
			return getModel().getFirstPointY();
		else
			return 0;
	}
	
	/** See {@link LineSegmentModel#getLength()}. */
	public int getLength() {
		if (getModel() != null)
			return getModel().getLength();
		else
			return 0;
	}

	/** See {@link LineSegmentModel#getSecondPoint()}. */
	public Point getSecondPoint() {
		if (getModel() != null)
			return getModel().getSecondPoint();
		else
			return null;
	}

	/** See {@link LineSegmentModel#getSecondPointX()}. */
	public int getSecondPointX() {
		if (getModel() != null)
			return getModel().getSecondPointX();
		else
			return 0;
	}
	
	/** See {@link LineSegmentModel#getSecondPointY()}. */
	public int getSecondPointY() {
		if (getModel() != null)
			return getModel().getSecondPointY();
		else
			return 0;
	}
	
	/** See {@link LineSegmentModel#getFirstPoint()}. */
	public void setFirstPoint(Point firstPoint) {
		if (getModel() != null)
			getModel().setFirstPoint(firstPoint);
	}

	/** See {@link LineSegmentModel#setFirstPointX(int)}. */
	public void setFirstPointX(int x) {
		if (getModel() != null)
			getModel().setFirstPointX(x);
	}
	
	/** See {@link LineSegmentModel#setFirstPointX(int)}. */
	public void setFirstPointY(int y) {
		if (getModel() != null)
			getModel().setFirstPointY(y);
	}
	
	/** See {@link LineSegmentModel#getFirstPoint()}. */
	public void setSecondPoint(Point secondPoint) {
		if (getModel() != null)
			getModel().setSecondPoint(secondPoint);
	}
	
	/** See {@link LineSegmentModel#setSecondPointX(int)}. */
	public void setSecondPointX(int x) {
		if (getModel() != null)
			getModel().setSecondPointX(x);
	}
	
	/** See {@link LineSegmentModel#setSecondPointX(int)}. */
	public void setSecondPointY(int y) {
		if (getModel() != null)
			getModel().setSecondPointY(y);
	}

}
