/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'FigureModelExtended.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.model;


import java.awt.Dimension;
import java.awt.Rectangle;

/**
 * <code>FigureModelExtended</code> extends {@link FigureModel} by providing
 * write access to the figure's bounds. This model is interesting for such
 * figures whose bounding box defines the figure such as a rectangle or square. 
 * 
 * @version 0.1 11/08/10
 * @author Christian Wolf
 * 
 */
public interface FigureModelExtended extends FigureModel {

	/**
	 * Moves (increases or decreases) the height of this model's figure.
	 * 
	 * @param dHeight
	 *            the delta-height to move the height by
	 * @see #moveWidthBy(int)
	 * @see #moveSizeBy(int, int)
	 * @see #moveSizeBy(Dimension)
	 */
	void moveHeightBy(int dHeight);

	/**
	 * Moves (increases or decreases) the size of this model's figure.
	 * 
	 * @param dSize
	 *            the <code>delta-size</code> to move the size by
	 * @see #moveSizeBy(int, int)
	 * @see #moveWidthBy(int)
	 * @see #moveHeightBy(int)
	 */
	void moveSizeBy(Dimension dSize);
	
	/**
	 * Moves (increases or decreases) the size of this model's figure.
	 * 
	 * @param dWidth
	 *            the <code>delta-width</code> to move the width by
	 * @param dHeight
	 *            the <code>delta-height</code> to move the height by
	 * @see #moveSizeBy(Dimension)
	 * @see #moveWidthBy(int)
	 * @see #moveHeightBy(int)
	 */
	void moveSizeBy(int dWidth, int dHeight);
	
	/**
	 * Moves (increases or decreases) the width of this model's figure.
	 * 
	 * @param dWidth
	 *            the <code>delta-width</code> to move the width by
	 * @see #moveHeightBy(int)
	 * @see #moveSizeBy(int, int)
	 * @see #moveSizeBy(Dimension)
	 */
	void moveWidthBy(int dWidth);

	/**
	 * Sets the bounds (size and location) of this model's figure.
	 * 
	 * @param x
	 *            the new x-location
	 * @param y
	 *            the new y-location
	 * @param width
	 *            the new width
	 * @param height
	 *            the new height
	 * @see #setBounds(Rectangle)
	 */
	void setBounds(int x, int y, int width, int height);
	


	/**
	 * Sets the bounds (size and location) of this model's figure.
	 * 
	 * @param bounds
	 *            the new bounds
	 * @see #setBounds(int, int, int, int)
	 */
	void setBounds(Rectangle bounds);
	
	/**
	 * Sets the height of the figure.
	 * 
	 * @param height
	 *            the new height
	 * @see #setWidth(int)
	 * @see #setSize(int, int)
	 * @see #setSize(Dimension)
	 */
	void setHeight(int height);
	
	/**
	 * Sets the size (width and height) of this model's figure.
	 * 
	 * @param size
	 *            the new size
	 * @see #setSize(int, int)
	 * @see #setHeight(int)
	 * @see #setWidth(int)
	 */
	void setSize(Dimension size);

	/**
	 * Sets the size (width and height) of this model's figure.
	 * 
	 * @param width
	 *            the new width
	 * @param height
	 *            the new height
	 * @see #setSize(Dimension)
	 * @see #setHeight(int)
	 * @see #setWidth(int)
	 */
	void setSize(int width, int height);
	
	/**
	 * Sets the width of this model's figure.
	 * 
	 * @param width
	 *            the new width
	 * @see #setHeight(int)
	 * @see #setSize(Dimension)
	 * @see #setSize(int, int)
	 */
	void setWidth(int width);
	
}
