/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'PolylineModel.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.model;

import java.awt.Point;

/**
 * <code>PolylineModel</code> is the interface for the data model of a polyline
 * figure. If the polyline figure is closed, i.e., the polyline has at least
 * three points and its first and last point have the same coordinates, it might
 * be treated as a polygon figure. Thus this model serves for both the related
 * figures. This interface is inherited from {@link FigureModel}. Besides the
 * information maintained by <code>FigureModel</code>, this model supports
 * access to the point set defining the poly[gon|line].
 * 
 * @version 0.1 11/08/10
 * @author Christian Wolf
 * 
 */
public interface PolylineModel extends FigureModel {

	/**
	 * Adds a point to this model's poly[gon|line].
	 * 
	 * @param x
	 *            the x-location of the new point
	 * @param y
	 *            the y-location of the new point
	 * @return true, if adding was successful, false otherwise
	 */
	boolean addPoint(int x, int y);

	/**
	 * Adds a point to this model's poly[gon|line] at the specified index.
	 * 
	 * @param index
	 *            the index the new point is added to
	 * @param x
	 *            the x-location of the new point
	 * @param y
	 *            the y-location of the new point
	 * @return true, if adding was successful, false otherwise
	 */
	boolean addPoint(int index, int x, int y);

	/**
	 * Adds a point to this model's poly[gon|line] at the specified index.
	 * 
	 * @param index
	 *            the index the new point is added to
	 * @param point
	 *            the new point
	 * @return true, if adding was successful, false otherwise
	 */
	boolean addPoint(int index, Point point);

	/**
	 * Adds a point to this model's poly[gon|line].
	 * 
	 * @param point
	 *            the new point
	 * @return true, if adding was successful, false otherwise
	 */
	boolean addPoint(Point point);

	/**
	 * Gets the index of the point of this model's poly[gon|line] with the given
	 * coordinates.
	 * 
	 * @param x
	 *            the x-location of the point
	 * @param y
	 *            the y-location of the point
	 * @return the current index
	 */
	int getIndexOfPoint(int x, int y);

	/**
	 * Gets the index of the point of this model's poly[gon|line].
	 * 
	 * @param point
	 *            the point whose index is computed
	 * @return the current index or -1 if point is not in the point set
	 */
	int getIndexOfPoint(Point point);

	/**
	 * Gets the total number of points defining this model's poly[gon|line].
	 * 
	 * @return the current total number of points
	 */
	int getNumberOfPoints();

	/**
	 * Gets the point of this model's poly[gon|line] with the given index.
	 * 
	 * @param index
	 *            the index of the point
	 * @return the point
	 */
	Point getPoint(int index);

	/**
	 * Gets all points of this model's poly[gon|line].
	 * 
	 * @return all points
	 */
	Point[] getPoints();

	/**
	 * Gets all x-locations of all points of this model's poly[gon|line].
	 * 
	 * @return all x-locations
	 * @deprecated use {@link #toAWTPolygon()} and then
	 *             <code>getXPoints()</code> instead
	 */
	int[] getXPoints();

	/**
	 * Gets all y-locations of all points of this model's poly[gon|line].
	 * 
	 * @return all y-locations
	 * @deprecated use {@link #toAWTPolygon()} and then
	 *             <code>getYPoints()</code> instead
	 */
	int[] getYPoints();

	/**
	 * Determines whether this model's polyline is closed. The polyline is
	 * closed and therewith is a polygon, if at least has three points whereof
	 * the first and the last point have the same coordinates.
	 * 
	 * @return true or false
	 */
	boolean isClosed();

	/**
	 * Removes the point with the given index from this model's poly[gon|line].
	 * 
	 * @param index
	 *            the index of the point to remove
	 * @return true, if removing was successful, false otherwise
	 */
	boolean removePoint(int index);

	/**
	 * Removes the point with the given coordinates from this model's
	 * poly[gon|line].
	 * 
	 * @param x
	 *            the x-location of the point
	 * @param y
	 *            the y-location of the point
	 * @return true, if removing was successful, false otherwise
	 */
	boolean removePoint(int x, int y);

	/**
	 * Removes the given point from this model's poly[gon|line].
	 * 
	 * @param point
	 *            the point to remove
	 * @return true, if removing was successful, false otherwise
	 */
	boolean removePoint(Point point);

	/**
	 * Substitutes the point with the given index by a new one.
	 * 
	 * @param index
	 *            the index of the point to be substituted
	 * @param x
	 *            the x-location
	 * @param y
	 *            the y-location
	 * @return the old point that was substituted
	 */
	Point setPoint(int index, int x, int y);

	/**
	 * Substitutes the point with the given index by a new one.
	 * 
	 * @param index
	 *            the index of the point to be substituted
	 * @param point
	 *            the new point
	 * @return the old point that was substituted
	 */
	Point setPoint(int index, Point point);

	/**
	 * Substitutes all the points of this model's poly[gon|line].
	 * 
	 * @param points
	 *            the new points
	 * @return true, if setting was successful, false otherwise
	 */
	boolean setPoints(Point[] points);

	/**
	 * 
	 * @return an AWT polygon with the point set of this model's poly[gon|line]
	 */
	java.awt.Polygon toAWTPolygon();

}
