/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'BasicFigureUI.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.plaf;

import geofasc.swing.AbstractFigure;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;

import javax.swing.JComponent;
import javax.swing.LookAndFeel;
import javax.swing.border.AbstractBorder;
import javax.swing.border.Border;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.PanelUI;
import javax.swing.plaf.basic.BasicPanelUI;


/**
 * Basic implementation of {@link FigureUI}.
 *
 * @version 0.1 08/11/10
 * @author Christian Wolf
 */
public class BasicFigureUI extends FigureUI {

	// Shared UI object
	private static PanelUI panelUI;

	public static ComponentUI createUI(JComponent c) {
		if (panelUI == null) {
			panelUI = new BasicPanelUI();
		}
		return panelUI;
	}

	/**
	 * Returns the baseline.
	 * 
	 * @throws NullPointerException
	 *             {@inheritDoc}
	 * @throws IllegalArgumentException
	 *             {@inheritDoc}
	 * @see javax.swing.JComponent#getBaseline(int, int)
	 * @since 1.6
	 */
	public int getBaseline(JComponent c, int width, int height) {
		super.getBaseline(c, width, height);
		Border border = c.getBorder();
		if (border instanceof AbstractBorder) {
			return ((AbstractBorder) border).getBaseline(c, width, height);
		}
		return -1;
	}

	/**
	 * Returns an enum indicating how the baseline of the component changes as
	 * the size changes.
	 * 
	 * @throws NullPointerException
	 *             {@inheritDoc}
	 * @see javax.swing.JComponent#getBaseline(int, int)
	 * @since 1.6
	 */
	public Component.BaselineResizeBehavior getBaselineResizeBehavior(
			JComponent c) {
		super.getBaselineResizeBehavior(c);
		Border border = c.getBorder();
		if (border instanceof AbstractBorder) {
			return ((AbstractBorder) border).getBaselineResizeBehavior(c);
		}
		return Component.BaselineResizeBehavior.OTHER;
	}

	protected void installDefaults(AbstractFigure f) {
		// don't change this

		LookAndFeel.installColorsAndFont(f, "Panel.background",
				"Panel.foreground", "Panel.font");
		LookAndFeel.installBorder(f, "Panel.border");
		LookAndFeel.installProperty(f, "opaque", Boolean.TRUE);

		// install here other properties

		if (f != null) {
			f.setOpaque(false);
			f.setFigureLineColor(Color.BLACK);
			f.setFigureFilled(false);
		}

	}

	/** {@inheritDoc} */
	@Override
	public void installUI(JComponent c) {
		AbstractFigure f = (AbstractFigure) c;
		super.installUI(f);
		installDefaults(f);
	}

	/** {@inheritDoc} */
	@Override
	public final void paint(Graphics g, JComponent c) {
		super.paint(g, c);
		paintFigure(g, (AbstractFigure) c);
	}

	/**
	 * Paints the given figure.
	 * 
	 * @param g the graphics context
	 * @param f the figure to paint
	 */
	protected void paintFigure(Graphics g, AbstractFigure f) {
		// just paint nothing here
	}

	protected void uninstallDefaults(AbstractFigure f) {
		LookAndFeel.uninstallBorder(f);
	}

	/** {@inheritDoc} */
	@Override
	public void uninstallUI(JComponent c) {
		AbstractFigure f = (AbstractFigure) c;
		uninstallDefaults(f);
		super.uninstallUI(c);
	}

}
