/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'Circle.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing;

import geofasc.swing.model.CircleModel;
import geofasc.swing.model.DefaultCircleModel;
import geofasc.swing.model.FigureModel;
import geofasc.swing.plaf.BasicEllipseUI;
import geofasc.swing.plaf.EllipseUI;


/**
 * <code>Circle</code> is the component used for visualizing a circle
 * figure. See {@link CircleModel} for more information on how a circle
 * figure is represented. This class is inherited from {@link Ellipse}.
 * 
 * @version 0.1 08/11/10
 * @author Christian Wolf
 * @see DefaultCircleModel
 * @see EllipseUI
 * @see BasicEllipseUI
 * 
 */
@SuppressWarnings("serial")
public class Circle extends Ellipse {

	/**
	 * Constructs a new <code>Circle</code> with radius 0.
	 */
	public Circle() {
		this(0);
	}

	/**
	 * Constructs a new <code>Circle</code> with the given <code>radius</code>.
	 * 
	 * @param diameter the initial diameter of this circle
	 */
	public Circle(int diameter) {
		this(0, 0, diameter);
	}

	/**
	 * Constructs a new <code>Circle</code> with the given <code>radius</code>
	 * at location <code>(x,y)</code> . 
	 *
	 * @param x the initial x-location
	 * @param y the initial y-location
	 * @param diameter the initial diameter
	 */
	public Circle(int x, int y, int diameter) {
		super(x, y, diameter, diameter);
	}

	/** {@inheritDoc} */
	@Override
	protected FigureModel createDefaultModel() {
		return new DefaultCircleModel();
	}
	
	/** {@inheritDoc} */
	@Override
	public CircleModel getModel() {
		return (CircleModel) super.getModel();
	}

	/**
	 * Sets the model of this circle.
	 * 
	 * @param model
	 *            the new model
	 */
	public void setModel(CircleModel model) {
		super.setModel(model);
	}
	
	// Convenience methods as syntactic sugar which just delegate
	// their calls to the model. Getter methods return the values
	// as returned by the model and might be null!
	
	/** See {@link CircleModel#getRadius()}. */
	public int getDiameter() {
		if (getModel() != null)
			return getModel().getDiameter();
		else return 0;
	}
	
	/** See {@link CircleModel#getRadius()}. */
	public int getRadius() {
		if (getModel() != null)
			return getModel().getRadius();
		else return 0;
	}
	
	/** See {@link CircleModel#setRadius(int)}. */
	public void setDiameter(int diameter) {
		if (getModel() != null)
			getModel().setRadius(diameter);
	}
	
	/** See {@link CircleModel#setRadius(int)}. */
	public void setRadius(int radius) {
		if (getModel() != null)
			getModel().setRadius(radius);
	}
	
}
