/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'TrafficLight.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.demo;

import geofasc.swing.Rectangle;

import java.awt.Color;

/**
 * A <code>TrafficLight</code> is a {@link Rectangle} with the common
 * functionality of a traffic light. It has three lamps of red, yellow and green
 * color and supports all known states. It comprises three {@link Led}s to
 * demonstrate how to build new components with geofasc. It especially shows
 * that a geofasc component like other Swing components/ containers can include
 * other components.
 * 
 * @author Christian Wolf
 * 
 */
@SuppressWarnings("serial")
public class TrafficLight extends Rectangle {

	public static enum TrafficLightState {
		RED, YELLOW, GREEN, RED_YELLOW
	};

	private Led mRedLamp;
	private Led mYellowLamp;
	private Led mGreenLamp;
	private TrafficLightState mState;

	public TrafficLight() {
		super(70, 190);
		setFigureFilled(true);
		setFigureFillColor(Color.BLACK);

		mRedLamp = new Led(10, 10, 50, Color.RED, Color.DARK_GRAY);
		mYellowLamp = new Led(10, 70, 50, Color.YELLOW, Color.DARK_GRAY);
		mGreenLamp = new Led(10, 130, 50, Color.GREEN, Color.DARK_GRAY);

		setState(TrafficLightState.RED);
		add(mRedLamp);
		add(mYellowLamp);
		add(mGreenLamp);
	}

	public TrafficLightState getState() {
		return mState;
	}

	public void setNextState() {
		switch (mState) {
		case RED:
			setState(TrafficLightState.RED_YELLOW);
			break;
		case YELLOW:
			setState(TrafficLightState.RED);
			break;
		case GREEN:
			setState(TrafficLightState.YELLOW);
			break;
		case RED_YELLOW:
			setState(TrafficLightState.GREEN);
			break;
		}
	}

	public void setState(TrafficLightState state) {
		mState = state;
		update();
	}

	private void update() {
		switch (mState) {
		case RED:
			mRedLamp.setOn();
			mYellowLamp.setOff();
			mGreenLamp.setOff();
			break;
		case YELLOW:
			mRedLamp.setOff();
			mYellowLamp.setOn();
			mGreenLamp.setOff();
			break;
		case GREEN:
			mRedLamp.setOff();
			mYellowLamp.setOff();
			mGreenLamp.setOn();
			break;
		case RED_YELLOW:
			mRedLamp.setOn();
			mYellowLamp.setOn();
			mGreenLamp.setOff();
			break;
		}
	}

}
