/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'Pencil.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

package geofasc.swing.demo;

import geofasc.swing.AbstractFigure;
import geofasc.swing.Rectangle;
import geofasc.swing.plaf.RectangleUI;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Rectangle2D;

import javax.swing.UIManager;

/**
 * Ein <code>TwinFluegel</code> ist ein doppelter Flügel eines {@link Windrad}
 * es. Der Flügel kann sich um seine Rotorachse drehen. Die Drehung erfolgt
 * dabei nach mathematischer Definition entgegen dem Uhrzeigerlauf. Bei einem
 * Drehwinkel von 0° und 180° ist der Flügel waagerecht, bei 90° und 270°
 * senkrecht, bei anderen Drehwinkeln entsprechend (z.B. 45° diagonal).
 * 
 * @author Christian Wolf
 */
@SuppressWarnings("serial")
public class TwinFluegel extends Rectangle {

	public static abstract class TwinWindradFluegelUI extends RectangleUI {
	}

	public static class BasicTwinWindradFluegelUI extends TwinWindradFluegelUI {

		@Override
		protected void paintFigure(Graphics g, AbstractFigure f) {
			Graphics2D g2d = (Graphics2D) g;
			TwinFluegel fluegel = (TwinFluegel) f;

			Ellipse2D fluegelLinks = new Ellipse2D.Float(0, 70, 70, 10);
			Ellipse2D achse = new Ellipse2D.Float(70, 70, 10, 10);
			Ellipse2D fluegelRechts = new Ellipse2D.Float(80, 70, 70, 10);
			Rectangle2D generator = new Rectangle2D.Float(60, 65, 30, 20);

			AffineTransform at = AffineTransform.getTranslateInstance(0, 0);
			at.rotate(-1 * Math.toRadians(fluegel.winkel), 75, 75);

			g2d.setColor(Color.GRAY);
			g2d.fill(generator);

			g2d.setColor(Color.DARK_GRAY);
			g2d.fill(at.createTransformedShape(fluegelLinks));
			g2d.fill(at.createTransformedShape(fluegelRechts));

			g2d.setColor(Color.BLACK);
			g2d.fill(at.createTransformedShape(achse));

		}

	}

	// Attribute
	
	/**
	 * Drehwinkel des Flügels mit mathematisch positiver Orientierung in Grad
	 * [°]
	 */
	private double winkel;

	// Konstruktoren
	
	/**
	 * Erzeugt ein neues TwinFluegel-Objekt.
	 */
	public TwinFluegel() {
		this(0);
	}

	/**
	 *  Erzeugt ein neues TwinFluegel-Objekt mit übergebenem Drehwinkel besitzt.
	 */
	public TwinFluegel(double pWinkel) {
		this(0, 0, pWinkel);
	}

	/**
	 * Erzeugt ein neues Windrad-Objekt mit übergebener x-y-Position und
	 * übergebenem Drehwinkel.
	 */
	public TwinFluegel(int x, int y, double pWinkel) {
		super(x, y, 150, 150);
		setWinkel(pWinkel);
	}

	// Methoden
	
	/**
	 * Gibt den Drehwinkel (zwischen 0° und 360°) dieses Flügels zurück.
	 */
	public double getWinkel() {
		return winkel;
	}

	/**
	 * Setzt den Drehwinkel (zwischen 0° und 360°) dieses Flügels.
	 */
	public void setWinkel(double pWinkel) {
		winkel = Math.abs(pWinkel) % 360;
		getModel().setRoundedCorners(!getModel().hasRoundedCorners());
	}

	/**
	 * Dreht diesen Flügel um den angegebenen Winkel.
	 */
	public void dreheUm(double pDWinkel) {
		setWinkel(winkel + Math.abs(pDWinkel));
	}

	/**
	 * Sets the look and feel (L&F) object that renders this component.
	 * 
	 * @param ui
	 *            the new L&F object
	 */
	public void setUI(TwinWindradFluegelUI ui) {
		super.setUI(ui);
	}

	/** {@inheritDoc} */
	@Override
	public void updateUI() {
		if (UIManager.get(getUIClassID()) != null) {
			setUI((TwinWindradFluegelUI) UIManager.getUI(this));
		} else {
			setUI(new BasicTwinWindradFluegelUI());
		}
	}

}
