/*
 * Copyright 2010 Christian Wolf, all rights reserved.
 * 
 * This file 'LineSegmentModel.java' is part of geofasc.
 * 
 * geofasc is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * geofasc is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
package geofasc.swing.model;


import java.awt.Point;

/**
 * <code>LineSegmentModel</code> is the interface for the data model of a line
 * segment figure. This interface is inherited from {@link FigureModel}. </p>
 * 
 * A line is defined by its two restricting points.
 * 
 * @version 0.1 11/08/10
 * @author Christian Wolf
 * 
 */
public interface LineSegmentModel extends FigureModel {

	/**
	 * Gets the first point of this model's line segment (the coordinates of the
	 * point refer to the coordinate system of this model's line segment).
	 * 
	 * @return the current first point
	 * @see #getFirstPointX()
	 * @see #getFirstPointY()
	 */
	Point getFirstPoint();

	/**
	 * Gets the x-location of the first point of this model's line segment (the
	 * coordinate refers to the coordinate system of this model's line segment).
	 * 
	 * @return the current x-location
	 * @see #getFirstPointY()
	 * @see #getFirstPoint()
	 */
	int getFirstPointX();

	/**
	 * Gets the y-location of the first point of this model's line segment (the
	 * coordinate refers to the coordinate system of this model's line segment).
	 * 
	 * @return the current y-location
	 * @see #getFirstPointX()
	 * @see #getFirstPoint()
	 */
	int getFirstPointY();

	/**
	 * Gets the length of this model's line segment.
	 * 
	 * @return the current length
	 */
	int getLength();

	/**
	 * Gets the second point of this model's line segment (the coordinates of
	 * the point refer to the coordinate system of this model's line segment).
	 * 
	 * @return the current second point
	 * @see #getSecondPointX()
	 * @see #getSecondPointY()
	 */
	Point getSecondPoint();

	/**
	 * Gets the x-location of the second point of this model's line segment (the
	 * coordinate refers to the coordinate system of this model's line segment).
	 * 
	 * @return the current x-location
	 * @see #getSecondPointY()
	 * @see #getSecondPoint()
	 */
	int getSecondPointX();

	/**
	 * Gets the y-location of the second point of this model's line segment (the
	 * coordinate refers to the coordinate system of this model's line segment).
	 * 
	 * @return the current y-location
	 * @see #getSecondPointX()
	 * @see #getSecondPoint()
	 */
	int getSecondPointY();

	/**
	 * Sets the first point of this model's line segment (the coordinates refer
	 * to the coordinate system of this model's line segment).
	 * 
	 * @param x
	 *            the new x-location
	 * @param y
	 *            the new y-location
	 * @see #setFirstPoint(Point)
	 * @see #setFirstPointX(int)
	 * @see #setFirstPointY(int)
	 */
	void setFirstPoint(int x, int y);

	/**
	 * Sets the first point of this model's line segment (the coordinates of the
	 * point refer to the coordinate system of this model's line segment).
	 * 
	 * @param firstPoint
	 *            the new first point
	 * @see #setFirstPoint(int, int)
	 * @see #setFirstPointX(int)
	 * @see #setFirstPointY(int)
	 */
	void setFirstPoint(Point firstPoint);

	/**
	 * Sets the x-location of the first point of this model's line segment (the
	 * coordinate refers to the coordinate system of this model's line segment).
	 * 
	 * @param x
	 *            the new x-location
	 * @see #setFirstPointY(int)
	 * @see #setFirstPoint(Point)
	 * @see #setFirstPoint(int, int)
	 */
	void setFirstPointX(int x);

	/**
	 * Sets the y-location of the first point of this model's line segment (the
	 * coordinate refers to the coordinate system of this model's line segment).
	 * 
	 * @param y
	 *            the new y-location
	 * @see #setFirstPointX(int)
	 * @see #setFirstPoint(Point)
	 * @see #setFirstPoint(int, int)
	 */
	void setFirstPointY(int y);

	/**
	 * Sets the second point of this model's line segment (the coordinates refer
	 * to the coordinate system of this model's line segment).
	 * 
	 * @param x
	 *            the new x-location of the second point
	 * @param y
	 *            the new y-location of the second point
	 * @see #setSecondPoint(Point)
	 * @see #setSecondPointX(int)
	 * @see #setSecondPointY(int)
	 */
	void setSecondPoint(int x, int y);

	/**
	 * Sets the second point of this model's line segment (the coordinates of
	 * the point refer to the coordinate system of this model's line segment).
	 * 
	 * @param secondPoint
	 *            the new second point
	 * @see #setSecondPoint(int, int)
	 * @see #setSecondPointX(int)
	 * @see #setSecondPointY(int)
	 */
	void setSecondPoint(Point secondPoint);

	/**
	 * Sets the x-location of the second point of this model's line segment (the
	 * coordinate refers to the coordinate system of this model's line segment).
	 * 
	 * @param x
	 *            the new x-location
	 * @see #setSecondPointY(int)
	 * @see #setSecondPoint(Point)
	 * @see #setSecondPoint(int, int)
	 */
	void setSecondPointX(int x);

	/**
	 * Sets the y-location of the second point of this model's line segment (the
	 * coordinate refers to the coordinate system of this model's line segment).
	 * 
	 * @param y
	 *            the new y-location
	 * @see #setSecondPointX(int)
	 * @see #setSecondPoint(Point)
	 * @see #setSecondPoint(int, int)
	 */
	void setSecondPointY(int y);

}
