package geofasc.swing.tool;

import java.util.Random;

/**
 * A <code>Calculator</code> provides easy access for often used mathematical
 * operations like computing the square of two numbers or generating a random
 * number.
 * 
 * @author Christian Wolf
 * @version 0.1 20/03/11
 */
public class Calculator {

	private static int sSeedOffset;
	private Random mRandom;

	/**
	 * Creates a new <code>Calculator</code>.
	 */
	public Calculator() {

		if (sSeedOffset == 0)
			mRandom = new Random();
		else
			mRandom = new Random(System.currentTimeMillis() + sSeedOffset);

		sSeedOffset++;
	}

	/**
	 * See {@link Math#abs(double)}.
	 */
	public double abs(double num) {
		return Math.abs(num);
	}

	/**
	 * See {@link Math#abs(int)}.
	 */
	public int abs(int num) {
		return Math.abs(num);
	}

	/**
	 * Computes the integer part of the given double number.
	 * 
	 * @param num
	 *            the double number whose integer part is to compute
	 * @return the integer part
	 */
	public int intPart(double num) {
		return (int) num;
	}

	/**
	 * Returns the value of the first argument raised to the power of the second
	 * argument.
	 * 
	 * @param base
	 *            the base (has to be non-null)
	 * @param exp
	 *            the exponent
	 * @return the power
	 */
	public double power(double base, double exp) {
		return Math.exp(Math.log(base) * exp);
	}

	/**
	 * Returns the value of the first argument raised to the power of the second
	 * argument.
	 * 
	 * @param base
	 *            the base (has to be non-null)
	 * @param exp
	 *            the exponent
	 * @return the power
	 */
	public int power(int base, int exp) {
		return (int) Math.round(Math.exp(Math.log(base) * exp));
	}
	
	/**
	 * Generates a random double number.
	 * 
	 * @return the random number
	 */
	public double random() {
		return mRandom.nextDouble();
	}

	/**
	 * Generates a random integer.
	 * 
	 * @return the random number
	 */
	public int randomInt() {
		return mRandom.nextInt();
	}

	/**
	 * Generates a random integer between the given intervall (including the
	 * left and right bounds of the intervall).
	 * 
	 * @param fromNum
	 *            the left number of the intervall
	 * @param toNum
	 *            the right number of the intervall
	 * @return the random number
	 */
	public int randomInt(int fromNum, int toNum) {
		return Math.abs(mRandom.nextInt()) % (toNum - fromNum + 1) + fromNum;
	}

	/**
	 * See {@link Math#round(double)}.
	 */
	public int round(double num) {
		return (int) Math.round(num);
	}

	/**
	 * See {@link Math#sqrt(double)}.
	 */
	public double sqrt(double num) {
		return Math.sqrt(num);
	}

	/**
	 * Returns the square of the given double number.
	 * 
	 * @param num
	 *            the double number to square
	 * @return the square
	 */
	public double square(double num) {
		return num * num;
	}

	/**
	 * Returns the square of the given integer number.
	 * 
	 * @param num
	 *            the integer number to square
	 * @return the square
	 */
	public int square(int num) {
		return num * num;
	}

}
